using System;
using System.Diagnostics;

namespace Implab {
    class PromiseActionReaction<T> : PromiseReaction<T> {
        readonly Action<T> m_fulfilled;

        readonly Action<Exception> m_rejected;

        readonly Deferred m_next;

        public PromiseActionReaction(Action<T> fulfilled, Action<Exception> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = (x) => {
                    fulfilled(x);
                    next.Resolve();
                };

            if (rejected != null)
                m_rejected = (x) => {
                    rejected(x);
                    next.Resolve();
                };
            m_next = next;
        }

        public PromiseActionReaction(Func<T, IPromise> fulfilled, Func<Exception, IPromise> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = (x) => { next.Resolve(fulfilled(x)); };
            if (rejected != null)
                m_rejected = (e) => { next.Resolve(rejected(e)); };
            m_next = next;
        }

        public PromiseActionReaction(Action<T> fulfilled, Func<Exception, IPromise> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = (x) => {
                    fulfilled(x);
                    next.Resolve();
                };

            if (rejected != null)
                m_rejected = (e) => { next.Resolve(rejected(e)); };
            m_next = next;
        }

        public PromiseActionReaction(Func<T, IPromise> fulfilled, Action<Exception> rejected, Deferred next, IDispatcher dispatcher) : base(dispatcher) {
            if (fulfilled != null)
                m_fulfilled = (x) => { next.Resolve(fulfilled(x)); };

            if (rejected != null)
                m_rejected = (x) => {
                    rejected(x);
                    next.Resolve();
                };
            m_next = next;
        }

        protected override bool HasFulfilHandler => m_fulfilled != null;

        protected override bool HasRejectHandler => m_rejected != null;

        protected override void DefaultReject(Exception reason) {
            m_next.Reject(reason);
        }

        protected override void DefaultResolve(T result) {
            m_next.Resolve();
        }

        protected override void RejectImpl(Exception reason) {
            try {
                m_rejected(reason);
            } catch (Exception e) {
                m_next.Reject(e);
            }
        }

        protected override void ResolveImpl(T result) {
            try {
                m_fulfilled(result);
            } catch (Exception e) {
                m_next.Reject(e);
            }
        }
    }
}